/*************************************************************************
 * The contents of this file are subject to the MYRICOM MX AND GM-2      *
 * MAPPING SOFTWARE AND DOCUMENTATION LICENSE (the "License"); User may  *
 * not use this file except in compliance with the License.  The full    *
 * text of the License can found in mapper directory in LICENSE.TXT      *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include "lx.h"

#ifdef MI_SIMULATION
#define LX_TOTAL_HOSTS 200
#define LX_TOTAL_XBARS 100
#else
#define LX_TOTAL_HOSTS LX_MAX_HOSTS
#define LX_TOTAL_XBARS LX_MAX_XBARS
#endif

const char lx_tag_string[] = "@(#)$Name: mx-1_2_0g $";

static int
lx_check_host_alloc (lx_t*lx)
{
  int i, c;
  insist (lx);
  
  c = lx->free_hosts.count + lx->staged_hosts.count;
  
  for (i = 0; i < MI_MY_SIZE; i++)
    c += lx->maps [i].hosts.count;
  
  return c == LX_TOTAL_HOSTS - 1;
  
  except: return 0;
}

static int
lx_check_xbar_alloc (lx_t*lx)
{
  int i, c;
  insist (lx);
  
  c = lx->free_xbars.count + lx->staged_xbars.count;
  
  for (i = 0; i < MI_MY_SIZE; i++)
    c += lx->maps [i].frontier.count + lx->maps [i].xbars.count + lx->maps [i].merge_queue.count;
  
  return c == LX_TOTAL_XBARS;
  
  except: return 0;
}

lx_node_t*
lx_peek (lx_queue_t*q)
{
  insist (q);
  return q->head;
  except: return 0;
}

lx_node_t*
lx_get (lx_queue_t*q)
{
  insist (q);
  if (q->head)
    return lx_remove (q, q->head);
  except: return 0;
}

int
lx_queued (lx_queue_t*q, lx_node_t*n)
{
  insist (q && n);
  return n->q == q;
  except: return 0;
}

int
lx_put (lx_queue_t*q, lx_node_t*n)
{
  insist (q && n);
  insist (!n->next && !n->prev);
  insist (!n->q);

  if (q->tail)
  {
    q->tail->next = mi_node2index (n);
    n->prev = mi_node2index (q->tail);
  }
  else
  {
    insist (!q->head);
    q->head = n;
    n->prev = 0;
  }
  q->tail = n;
  n->next = 0;
  n->q = q;
  q->count++;
  
  return 1;
  except: return 0;
}

int
lx_put_at_front (lx_queue_t*q, lx_node_t*n)
{
  insist (q && n);
  insist (!n->next && !n->prev);
  insist (!n->q);

  if (q->head)
  {
    n->next = mi_node2index (q->head);
    q->head->prev = mi_node2index (n);
  }
  else
  {
    insist (!q->tail);
    q->tail = n;
    n->prev = 0;
  }
  q->head = n;
  n->prev = 0;
  n->q = q;
  q->count++;
  
  return 1;
  except: return 0;
}

int
lx_insert (lx_queue_t*q, lx_node_t*before, lx_node_t*n)
{
  insist (q && n);
  insist (!n->next && !n->prev);
  insist (!n->q);
  
  if (q->tail)
  {
    insist (q->head);
    insist (q->head != n);
    
    if (!before)
    {
      n->next = mi_node2index (q->head);
      q->head->prev = mi_node2index (n);
      q->head = n;
      n->prev = 0;
    }
    else if (before == q->tail)
      return lx_put (q, n);
    else
    {
      mi_index2node (before->next)->prev = mi_node2index (n);
      n->next = before->next;
      before->next = mi_node2index (n);
      n->prev = mi_node2index (before);
    }
  }
  else
  {
    insist (!q->head);
    insist (!before);
    q->head = q->tail = n;
    n->prev = n->next = 0;
  }

  q->count++;
  n->q = q;

  return 1;
  except: return 0;
}

lx_node_t *
lx_remove (lx_queue_t*q, lx_node_t*n)
{
  lx_node_t*nn,*pp;
  
  insist (n);
  insist (n->q == q);
  n->q = 0;
  
  nn = mi_index2node (n->next);
  pp = mi_index2node (n->prev);

  insist (q->head && q->tail);
  
  if (nn)
    nn->prev = mi_node2index (pp);
  if (pp)
    pp->next = mi_node2index (nn);
  
  if (n == q->tail)
    q->tail = pp;
  if (n == q->head)
    q->head = nn;
  
  n->prev = n->next = 0;
  q->count--;
  
  return n;
  
  except: return 0;
}


lx_node_t *
lx_remove_from_anywhere (lx_node_t*n)
{
  insist (n);
  return n->q ? lx_remove (n->q, n) : n;
  except: return 0;
}

lx_node_t*lx_new_xbar (lx_map_t*m, int iport, lx_route_t*route, int id, int mask, int absolute_port)
{
  lx_node_t*n;
  
  insist (m && lx_check_xbar_alloc (m->lx));
  insist (iport >= 0 && iport < MI_MY_SIZE);
  
  n = (lx_node_t*) lx_get (&m->lx->free_xbars);

  insist (m && n);
  lx_bzero ((char*)n, sizeof (lx_xbar_t));
  
  n->xbar = 1;
  n->first = 0;
  lx_xbar_c (n)->id = id;
  n->index = m->num_xbars++;
  n->original.index = 0;
  n->original.port = 0;
  lx_xbar_c (n)->id = id;
  lx_xbar_c (n)->mask = mask;
  lx_xbar_c (n)->absolute_port = absolute_port;
  
  if (route)
  {
    lx_xbar_c (n)->routes [iport].length = route->length;
    lx_copy_route (lx_xbar_c (n)->routes [iport].hops, route->hops,
                   lx_xbar_c (n)->routes [iport].length);
  }

  mi_c (("created " lx_node_format, lx_node_args (n)));
  
  insist (mi_index2node (mi_node2index (n)) == n);

  return n;
  except: return 0;  
}

lx_node_t *
lx_new_host (lx_map_t*m, int iport, int oiport, unsigned char mac_address [6], int host_type, int level, lx_route_t*route)
{
  lx_node_t*n;

  insist (m && lx_check_host_alloc (m->lx));
  insist (iport >= 0 && iport < MI_MY_SIZE);
  insist (oiport >= 0 && oiport < LX_HOST_SIZE);

  n = (lx_node_t*) lx_get (&m->lx->free_hosts);

  insist (m && n);
  lx_bzero ((char*)n, sizeof (lx_host_t));
  n->first = 0;
  n->index = m->num_hosts++;
  n->original.index = 0;
  n->original.port = 0;  

  lx_host_c (n)->level = level;
  lx_host_c (n)->host_type = host_type;

  if (mac_address)
    mi_memcpy (lx_host_c (n)->mac_address, mac_address, 6);
  
  if (route)
  {
    lx_host_c (n)->routes [iport] [oiport].length = route->length;
    lx_copy_route (lx_host_c (n)->routes [iport] [oiport].hops, route->hops, lx_host_c (n)->routes [iport] [oiport].length);
  }
  mi_c (("created " lx_node_format, lx_node_args (n)));

  insist (mi_index2node (mi_node2index (n)) == n);

  return n;
  except: return 0;
}

int lx_get_xbar_count (lx_node_t*n)
{
  int i;
  int c = 0;
  lx_node_t*nn;
  
  insist (n && n->xbar);

  for (i = 0; i < LX_XBAR_SIZE; i++)
  {
    if ((nn = lx_get_node (n, i + n->first)) && nn->xbar)
      c++;
  }
  return c;
  except: return 0;
}

void
lx_delete_hosts (lx_map_t*m, lx_node_t*n)
{
  int i;
  lx_node_t*nn;
  
  insist (m && n && n->xbar);

  for (i = 0; i < LX_XBAR_SIZE; i++)
  {
    if ((nn = lx_get_node (n, i + n->first)) && !nn->xbar)
      lx_delete_node (m, lx_remove_from_anywhere (nn));
  }
  except:;
}

int
lx_has_hosts (lx_node_t*n)
{
  int i;
  lx_node_t*o;
  insist (n->xbar);
  
  for (i = 1 - LX_XBAR_SIZE; i < LX_XBAR_SIZE; i++)
  {
    if ((o = lx_get_node (n, i)) && !o->xbar)
      return 1;
  }
  except: return 0;
}

void
lx_delete_node (lx_map_t*m, lx_node_t*n)
{
  int i;
  lx_node_t*o;
  
  insist (m && n);

  mi_c (("deleted " lx_node_format, lx_node_args (n)));
  
  for (i = 1 - LX_XBAR_SIZE; i < LX_XBAR_SIZE; i++)
  {
    mi_punt (m->lx);

    if ((o = lx_get_node (n, i)))
    {
      if (lx_get_node (o, lx_get_port (n, i)) == n)
	lx_disconnect (o, lx_get_port (n, i));
    }
  }  
  
  if (n->xbar)
    m->num_xbars--;
  else
    m->num_hosts--;

  insist (m->num_hosts >= 0 && m->num_xbars >= 0);

  lx_put (n->xbar ? &m->lx->free_xbars : &m->lx->free_hosts, n);

  except:;
}

static void
lx_shift (lx_node_t*n, unsigned amount)
{
  int i;
  
  insist (n && amount < LX_XBAR_SIZE);
  for (i = LX_XBAR_SIZE - 1; i >= (int) amount; i--)
    n->links [i] = n->links [i - amount];
  for (; i >= 0; i--)
    n->links [i].index = 0;
  except:;
}

void
lx_disconnect (lx_node_t*n, int p)
{
  insist (n);
  insist (p > -LX_XBAR_SIZE && p < LX_XBAR_SIZE);
  insist (p - n->first >= 0 && p - n->first < LX_XBAR_SIZE);
  n->links [p - n->first].index = 0;
  except:;
}


int
lx_connect (lx_node_t*na, int pa, lx_node_t*nb, int pb)
{
  insist (na && nb);
  insist (pa > -LX_XBAR_SIZE && pa < LX_XBAR_SIZE);
  insist (pb > -LX_XBAR_SIZE && pb < LX_XBAR_SIZE);
  insist (na->xbar || (pa >= 0 && pa < 2));
  insist (nb->xbar || (pb >= 0 && pb < 2));
  
  mi_c (("connecting " lx_node_format " port %d to " lx_node_format " port %d",
         lx_node_args (na), pa, lx_node_args (nb), pb));

  if (pa < na->first)
  {
    lx_shift (na, na->first - pa);
    na->first = pa;
  }
  if (pb < nb->first)
  {
    lx_shift (nb, nb->first - pb);
    nb->first = pb;
  }
  
  na->links [pa - na->first].port = pb;  
  na->links [pa - na->first].index = mi_node2index (nb);
  
  nb->links [pb - nb->first].port = pa;  
  nb->links [pb - nb->first].index = mi_node2index (na);

  return 1;
  except: return 0;
}

lx_node_t *
lx_get_node (lx_node_t*n, int port)
{
  insist (n);
  insist (n->xbar || (n->first == 0));  

  /*
   * Validate the port number - for hosts, must be in 0..LX_HOST_SIZE-1.
   * for xbars, must be in -LX_XBAR_SIZE+1..LX_XBAR_SIZE-1 AND
   * must be >= the first known port, but by no more than LX_XBAR_SIZE-1
   */
  if ((!n->xbar && (port < 0 || port >= LX_HOST_SIZE)) ||
      (n->xbar &&
       (port <= -LX_XBAR_SIZE || port >= LX_XBAR_SIZE ||
        port - n->first < 0 || port - n->first >= LX_XBAR_SIZE)))
    return 0;
  
  return mi_index2node (n->links [port - n->first].index);
  
  except: return 0;
}

void
lx_copy_route (signed char*dst, signed char*src, int length)
{
  int i;
  insist (dst && src && length <= LX_ROUTE_SIZE);
  
  for (i = 0; i < length; i++)
  {
    insist (src [i] >= 1 - LX_XBAR_SIZE && src [i] < LX_XBAR_SIZE);
    dst [i] = src [i];
  }
  except:;
}

void
lx_reverse_route (signed char*dst, signed char*src, int length)
{
  int i;
  insist (dst && src && length <= LX_ROUTE_SIZE);

  for (i = 0; i < length; i++)
  {
    insist (src [length - 1 - i] >= 1 - LX_XBAR_SIZE);
    insist (src [length - 1 - i] < LX_XBAR_SIZE);
    dst [i] = - src [length - 1 - i];
  }

  except:;
}

int lx_there_and_back_again (signed char*dst, signed char*src, signed char port, int length)
{
  insist (dst && src && length * 2 + 3 <= LX_ROUTE_SIZE);
 
  lx_copy_route (dst, src, length);
  dst [length] = port;
  dst [length + 1] = 0;
  dst [length + 2] = -port;
  
  lx_reverse_route (dst + length + 3, src, length);
  
  return length * 2 + 3;
  except: return 0;
}

void
lx_physical_route (signed char*p, int length)
{
  int i;
  insist (p && length <= LX_ROUTE_SIZE);
  
  for (i = 0; i < length; i++)
    p[i] = (p[i] & '\x3f') | '\x80';

  except:;
}

void
lx_logical_route (signed char*p, int length)
{
  int i;
  
  insist (p && length <= LX_ROUTE_SIZE);

  for (i = 0; i < length; i++)
    if ((p [i] = (p[i] & 0x3f)) & 0x20)
      p [i] |= 0xf0;

  except:;
}

void
lx_compose_route (lx_route_t*dst, lx_route_t*a, signed char hop, lx_route_t*b)
{
  insist (dst && a && b);
  
  lx_append_route (dst, a, hop);
  lx_copy_route (dst->hops + dst->length, b->hops, b->length);
  dst->length += b->length;

  except:;
}

lx_route_t *
lx_append_route (lx_route_t*dst, lx_route_t*route, signed char port)
{
  insist (dst && route && route->length < LX_ROUTE_SIZE);
  insist (port >= 1 - LX_XBAR_SIZE && port < LX_XBAR_SIZE);

  lx_copy_route (dst->hops, route->hops, route->length);
  dst->hops [route->length] = port;
  dst->length = route->length + 1;

  return dst;
  except: return 0;;
}



char *
lx_print_route (signed char*p, int length)
{
  static char s [LX_ROUTE_SIZE * 4];
  char*t = s;
  int i;

  insist (s && p && length <= LX_ROUTE_SIZE);

  *s = 0;

  for (i = 0; i < length; i++)
    t += sprintf (t, "%d,", p [i]);

  if (t > s)
    t [-1] = 0;
  return s;
  except: return 0;
}

char *
lx_print_change (int change)
{
  static char s [100];
  *s = 0;
  switch (change)
  {
    case LX_CHANGED_HOST:
      sprintf (s, "changed host");
      break;
    case LX_MISSING_HOST:
      sprintf (s, "missing host");
      break;
    case LX_MISSING_LINK:
      sprintf (s, "missing link");
      break;
    case LX_NEW_LINK:
      sprintf (s, "new link");
      break;
    case LX_CHANGED_ORDER_IN_COLLECT:
      sprintf (s, "changed order in collect");
      break;
    case LX_CHANGED_PORT_IN_COLLECT:
      sprintf (s, "changed port in collect");
      break;
    case LX_FAILED_ROUTING:
      sprintf (s, "failed routing");
      break;
    case LX_SCOUT_NOT_IN_MAP:
      sprintf (s, "scout not in map");
      break;
    case LX_REQUEST_NOT_IN_MAP:
      sprintf (s, "request not in map");
      break;
    case LX_SCOUT_BAD_PARENT:
      sprintf (s, "scout bad parent");
      break;
    default:
      sprintf (s, "%d", change);
      break;

  }
  return s;
}

void
lx_bzero (char*p, unsigned length)
{
  unsigned i;
  for (i = 0; i < length; i++)
    p [i] = 0;
}

int
lx_memcmp (unsigned char*a, unsigned char*b, unsigned length)
{
  unsigned i;
  for (i = 0; i < length; i++)
  {
    if (a [i] < b [i])
      return -1;
    else if (a [i] > b [i])
      return 1;
  }
  return 0;
}

int
lx_zero (unsigned char*p, unsigned length)
{
  unsigned i;
  for (i = 0; i < length; i++)
    if (p [i]) return 0;
  return 1;
}

void
lx_reset_map (lx_t*lx, lx_map_t*m)
{
  lx_node_t*n;
  
  insist (lx && m);
  insist (!m->map_valid);

  m->lx = lx;
  
  while ((n = (lx_node_t*) lx_get (&m->hosts)))
  {
    lx_put (&lx->free_hosts, n);
    mi_punt (lx);
  }
  
  while ((n = (lx_node_t*) lx_get (&m->xbars)))
  {
    lx_put (&lx->free_xbars, n);
    mi_punt (lx);
  }

  while ((n = (lx_node_t*) lx_get (&m->frontier)))
  {
    lx_put (&lx->free_xbars, n);
    mi_punt (lx);
  }
  
  insist (!lx_get (&m->merge_queue));
  
  m->root = 0;
  m->num_hosts = m->num_xbars = 0;
  m->routing_flag = lx->routing_flag;
  m->xbar_verify_flag = lx->xbar_verify_flag;
  m->seek_size = lx->seek_size;

  except:;
}


void
lx_init (lx_t*lx)
{
  int i;
  
  insist (lx);  

  lx_bzero ((char*)&lx->free_hosts, sizeof (lx_queue_t));
  lx_bzero ((char*)&lx->free_xbars, sizeof (lx_queue_t));
  
  lx_bzero ((char*)&lx->host_block, sizeof (lx_host_t) * LX_TOTAL_HOSTS);
  lx_bzero ((char*)&lx->xbar_block, sizeof (lx_xbar_t) * LX_TOTAL_XBARS);

  lx_bzero ((char*)&lx->staged_hosts, sizeof (lx_queue_t));
  lx_bzero ((char*)&lx->staged_xbars, sizeof (lx_queue_t));

  for (i = 0; i < MI_MY_SIZE; i++)
  {
    lx_bzero ((char*)&lx->maps [i].xbars, sizeof (lx_queue_t));
    lx_bzero ((char*)&lx->maps [i].hosts, sizeof (lx_queue_t));
    lx_bzero ((char*)&lx->maps [i].frontier, sizeof (lx_queue_t));
    lx_bzero ((char*)&lx->maps [i].merge_queue, sizeof (lx_queue_t));
    lx_bzero ((char*)&lx->maps [i].host_array,
    	      sizeof (lx->maps [i].host_array));
    lx_bzero ((char*)&lx->maps [i].xbar_array,
    	      sizeof (lx->maps [i].xbar_array));
    lx->maps [i].map_valid = 0;
    lx_bzero ((char*) lx->maps [i].map_address, 6);
    lx_bzero ((char*) lx->maps [i].winner_address, 6);
    lx_reset_map (lx, &lx->maps [i]);
    lx->changes [i].pending = 0;
  }  

  lx->maxes [LX_SMALL_SEND] = LX_NUM_SMALLS;
  lx->maxes [LX_BIG_SEND] = LX_NUM_BIGS;
  lx->maxes [LX_REPLY_SEND] = LX_NUM_REPLIES;

  for (i = 0; i < LX_NUM_SEND_TYPES; i++)
  {
    lx->sends [i] = 0;
    lx->pendings [i] = lx->maxes [i];
  }
  
  for (i = 0; i < LX_NUM_SMALLS; i++)
  {
    ((lx_buffer_t*) &lx->small_block [i])->type = LX_SMALL_SEND;
    lx_free_send (lx, (lx_buffer_t*) &lx->small_block [i]);
  }
  
  for (i = 0; i < LX_NUM_BIGS; i++)
  {
    ((lx_buffer_t*) &lx->big_block [i])->type = LX_BIG_SEND;
    lx_free_send (lx, (lx_buffer_t*) &lx->big_block [i]);
  }
  
  for (i = 0; i < LX_NUM_REPLIES; i++)
  {
    ((lx_buffer_t*) &lx->reply_block [i])->type = LX_REPLY_SEND;
    lx_free_send (lx, (lx_buffer_t*) &lx->reply_block [i]);
  }
  
  lx->duration = 0;
  lx->die = 0;
  lx->in_receive = 0;  
  lx->paused = lx->net_paused = 0;  
  
  for (i = 1; i < LX_TOTAL_HOSTS; i++)
  {
    lx_node_t*n = (lx_node_t*) &lx->host_block [i];
    n->xbar = 0;
    lx_put (&lx->free_hosts, n);
  }
  for (i = 0; i < LX_TOTAL_XBARS; i++)
  {
    lx_node_t*n = (lx_node_t*) &lx->xbar_block [i];
    n->xbar = 1;
    lx_put (&lx->free_xbars, n);
  }
  
  except:;
}

lx_node_t *
lx_seen (lx_map_t*m, unsigned char mac_address [6])
{
  lx_node_t*n;
  insist (m && mac_address);
  
  for (n = m->hosts.head; n; n = mi_index2node (n->next))
  { 
    if (!lx_memcmp (lx_host_c (n)->mac_address, mac_address, 6))
      return n;
    mi_punt (m->lx);
  }
  
  except: return 0;
}

int
lx_number_xbars_bfs (lx_map_t*m, int iport)
{
  lx_queue_t q;
  lx_node_t*n, *nn;
  int xbars_found, i;
  
  insist (m && m->root && m->host_array [0]);
  
  xbars_found = 0;
  
  for (i = 0, n = m->xbars.head; n; n = mi_index2node (n->next), i++)
  {
    mi_punt (m->lx);
    lx_xbar_c (n)->level = (unsigned char) -1;
  }

  insist (i == m->num_xbars);
  
  n = 0;
  if (iport < 0)
  {
    for (i = 0; i < LX_HOST_SIZE; i++)
      if ((n = lx_get_node (mi_index2node (m->host_array [0]), i)) && n->xbar)
	break;
  }
  else n = lx_get_node (m->root, iport);
  
  if (!n || !n->xbar)
    return 1;

  lx_bzero ((char*) &q, sizeof (q));
  
  lx_xbar_c (n)->level = 0;

  lx_put (&q, lx_remove (&m->xbars, n));
  
  while ((n = lx_get (&q)))
  {
    lx_put (&m->xbars, n);
  
    for (i = 0; i < LX_XBAR_SIZE; i++)
    { 
      mi_punt (m->lx);
      if ((nn = lx_get_node (n, i + n->first)))
      {
	if (nn->xbar)
	{
	  if (lx_xbar_c (nn)->level == (unsigned char) -1)
	  {
	    lx_xbar_c (nn)->level = lx_xbar_c (n)->level + 1;
	    lx_put (&q, lx_remove (&m->xbars, nn));
	  }
	}
	else
	  nn->original.port = lx_xbar_c (n)->level + 1;
      }
    }
  }
  
  insist (!q.count);
  return 1;
  except: return 0;
}

lx_node_t *
lx_bsearch_host (lx_map_t*m, unsigned char mac_address [6], int level)
{
  lx_node_t*n;
  int r, a, b, t;
  a = 0;
  b = m->num_hosts - 1;
  
  while (a <= b)
  {
    t = (a + b) / 2;
    n = mi_index2node (m->host_array [t]);
    insist (n);

    r = lx_hostcmp (lx_host_c (n)->mac_address,
                    lx_host_c (n)->level,
		    mac_address,
		    level);
    if (r == 0)
      return n;

    if (r > 0)
      a = t + 1;
    else
      b = t - 1;
  }
  
  except: return 0;
}

int
lx_before_sending (lx_map_t*m)
{
  int i;
  lx_node_t*n;
  
  insist (m && !m->map_valid);

  n = (lx_node_t*) lx_peek (&m->hosts);
  for (i = 0; n; i++, n = mi_index2node (n->next))
  {
    insist (!n->next ||
            lx_hostcmp (lx_host_c (n)->mac_address,
	                lx_host_c (n)->level,
			lx_host_c (mi_index2node (n->next))->mac_address,
			lx_host_c (mi_index2node (n->next))->level) > 0);
    
    m->host_array [i] = mi_node2index (n);
    n->index = i;
    mi_punt (m->lx);
  }

  insist (i == m->num_hosts);

  n = (lx_node_t*) lx_peek (&m->xbars);
  for (i = 0; n; i++, n = mi_index2node (n->next))
  {
    m->xbar_array [i] = mi_node2index (n);
    n->index = i;
    if (lx_xbar_c (n)->id)
      m->has_x32s = 1;
    
    mi_punt (m->lx);
  }
  insist (i == m->num_xbars);

  return 1;
  except: return 0;
}

void
lx_before_receiving (lx_map_t*m, int num_hosts, int num_xbars)
{
  int i;
  lx_node_t*n;
  
  insist (m);
  insist (m->num_hosts == 0 && m->num_xbars == 0);
  insist (!m->map_valid);
  
  for (i = 0; i < num_hosts; i++)
  {
    n = lx_new_host (m, 0, 0, 0, 0, 0, 0);
    insist (n);
    m->host_array [i] = mi_node2index (n);
    n->index = i;
    lx_put (&m->lx->staged_hosts, n);
    mi_punt (m->lx);
  } 
  insist (i == m->num_hosts);
  
  for (i = 0; i < num_xbars; i++)
  {
    n = lx_new_xbar (m, 0, 0, 0, 0, 0);
    insist (n);

    m->xbar_array [i] = mi_node2index (n);
    n->index = i;
    lx_put (&m->lx->staged_xbars, n);
    mi_punt (m->lx);
  }
  insist (i == m->num_xbars);
  except:;
}

int d_lates;

int
lx_compare_time (lx_t*lx, unsigned received, unsigned expected)
{
  insist (lx);

  if (received == expected)
  {
    lx->last_seen_timestamp = received;
    return 1;
  }
  
  if (received < expected && received != lx->last_seen_timestamp)
  {
    return 0;
    
    mi_c (("late message timestamp %u, expected %u", received, expected));
    if ((lx->timeout += lx->timeout / LX_TIMEOUT_INCREASE_FACTOR)
         > lx->start_timeout * LX_MAX_TIMEOUT_FACTOR)
      lx->timeout = lx->start_timeout * LX_MAX_TIMEOUT_FACTOR;
    
    mi_c (("timeout is now %u", lx->timeout));
    d_lates++;
  }
  except: return 0;
}

unsigned
lx_advance_time (lx_t*lx)
{
  insist (lx);
  mi_punt (lx);
  return ++lx->timestamp;
  except: return 0;
}


void
lx_set_timeout (lx_t*lx, unsigned timeout, unsigned tries)
{
  insist (lx);
  lx->timeout = lx->start_timeout = timeout;
  lx->tries = tries;
  
  except:;
}


int
lx_get_highest_mac_address (lx_map_t*m, unsigned char mac_address [6])
{
  int i;
  lx_node_t*n;
  
  insist (m && mac_address);
  
  if (!m->map_valid)
    return 0;

  n = (lx_node_t*) lx_peek (&m->hosts);
  for (i = 0; n; i++, n = mi_index2node (n->next))
  {
    if (lx_host_c (n)->host_type == m->lx->host_type)
    {
      mi_memcpy (mac_address, lx_host_c (n)->mac_address, 6);
      return 1;
    }
  }

  except: return 0;
}

void
lx_free_send (lx_t*lx, lx_buffer_t*p)
{
  insist (lx);
  insist (p);
  insist (p->type >= 0);
  insist (p->type < LX_NUM_SEND_TYPES);
  insist (lx->pendings [p->type] > 0);
  insist (lx->pendings [p->type] <= lx->maxes [p->type]);
  if (lx->pendings [p->type] != lx->maxes [p->type])
  {
    insist (lx->sends [p->type] != 0);
  }
  
  p->next = lx->sends [p->type];
  lx->sends [p->type] = p;
  lx->pendings [p->type]--;

  /*mi_c (("freed send type %d, %d pending", p->type, lx->pendings [p->type]));*/
  
  except:;
}

lx_buffer_t *
lx_alloc_send (lx_t*lx, int type)
{
  lx_buffer_t*p;

  insist (lx);
  insist (type >= 0);
  insist (type < LX_NUM_SEND_TYPES);
  
  if (!(p = lx->sends [type]))
    return 0;
  
  insist (lx->pendings [type] < lx->maxes [type]);
  insist (p->type == type);
  
  lx->sends [type] = p->next;
  lx->pendings [type]++;

  return p;
  except: return 0;
}

int
lx_hostcmp (unsigned char address1 [6],
            int level1,
	    unsigned char address2 [6],
	    int level2)
{
  insist (address1 && address2);
  return level1 == level2 ? lx_memcmp (address1, address2, 6) :
                            (level1 > level2 ? 1 : -1);
  except: return 0;
}

int
lx_higher_than_me (lx_t*lx, unsigned char mac_address [6], int level)
{
  insist (lx);
  insist (mac_address);
  return lx_hostcmp (mac_address, level, lx->mac_address, lx->level) > 0;
  except: return 0;
}

lx_node_t *
lx_follow (lx_node_t*n, int p, lx_route_t*r, int*last_port)
{ 
  int i;
  insist (n && r);
  
  insist (p >= 0 && p < LX_HOST_SIZE);
  insist (n && last_port);
  
  *last_port = lx_get_port (n, p);
  n = lx_get_node (n, p);
  
  for (i = 0; i < r->length; i++)
  {
    p = lx_get_port (n, *last_port + r->hops [i]);
    n = lx_get_node (n, *last_port + r->hops [i]);
    if (n == 0)
    {
      return 0;
    }
    *last_port = p;
  }
  return n;
  except: return 0;
}

int
lx_set_change (lx_t*lx,
               int iport,
	       unsigned char child_address [6],
	       unsigned timestamp,
	       int change,
	       unsigned map_version,
	       unsigned char map_address [6],
	       unsigned char leaf_address [6],
	       unsigned char mac_address [6],
	       int node,
	       int port,
	       unsigned xid
	       )
{
  lx_change_t*c = &lx->changes [iport];
  
  insist (lx);
  insist (iport >= 0);
  insist (iport < MI_MY_SIZE);
  insist (map_address);
  insist (leaf_address);
  
  c->map_version = map_version;

  if (child_address)
    mi_memcpy (c->child_address, child_address, 6);
  else
    lx_bzero ((char*) c->child_address, 6);
  c->timestamp = timestamp;
  mi_memcpy (c->map_address, map_address, 6);
  mi_memcpy (c->leaf_address, leaf_address, 6);
  if (mac_address)
    mi_memcpy (c->mac_address, mac_address, 6);
  else
    lx_bzero ((char*) c->mac_address, 6);
  
  c->node = node;
  c->port = port;
  c->change = change;
  c->pending = LX_REPORT_TRIES;
  c->xid = xid;
  
  mi_c (("\n\n@@@  ___________________________________________"));
  mi_c (("@@@  change is %s", lx_print_change (c->change)));
  mi_c (("@@@  child_address is " lx_mac_format,
  	 lx_mac_args (c->child_address)));
  mi_c (("@@@  timestamp is %d", c->timestamp));
  mi_c (("@@@  map version is %d", c->map_version));
  mi_c (("@@@  map_address is " lx_mac_format, lx_mac_args (c->map_address)));
  mi_c (("@@@  leaf_address is " lx_mac_format, lx_mac_args (c->leaf_address)));
  mi_c (("@@@  mac_address is " lx_mac_format, lx_mac_args (c->mac_address)));
  mi_c (("@@@  node is %d", c->node));
  mi_c (("@@@  port is %d", c->port));
  mi_c (("@@@  xid is %u", c->xid));
  mi_c (("@@@  -------------------------------------------\n\n"));
  return 1;
  except: return 0;
}



lx_node_t *
lx_find_host (lx_t*lx, lx_queue_t*queue, unsigned char mac_address [6], int level, lx_node_t**higher)
{
  int r;
  lx_node_t*n;
  insist (lx && queue && mac_address);
  insist (higher);
 
  *higher = 0;
  
  for (n = queue->head; n; n = mi_index2node (n->next))
  {
    if ((r = lx_hostcmp (lx_host_c (n)->mac_address, lx_host_c (n)->level, mac_address, level)) > 0)
      *higher = n;
    
    if (!r || !lx_memcmp (lx_host_c (n)->mac_address, mac_address, 6))
      return n;
    /*mi_punt (lx); do not ever punt here*/
  }

  except: return 0;
}

lx_node_t*lx_find_xbar (lx_t*lx, lx_queue_t*queue, int id)
{
  lx_node_t*n;
  insist (lx && queue && id);
 
  for (n = queue->head; n; n = mi_index2node (n->next))
    if (lx_xbar_c (n)->id == id)
      return n;

  except: return 0;
}

/*returns 1 if the relative port is in bounds and not known to be disconnected*/

int lx_in_bounds (lx_node_t*n, int port)
{
  insist (n && n->xbar);

  /*if it's an x16*/

  if (!lx_xbar_c (n)->id)
    return port >= 1 - LX_XBAR16_SIZE && port < LX_XBAR16_SIZE;
    
  /*it's an x32*/

  insist (lx_xbar_c (n)->absolute_port < LX_XBAR_SIZE);
 
  /*adjust to absolute*/
  port += lx_xbar_c (n)->absolute_port;
  
  return port >= 0 && port < LX_XBAR_SIZE && (lx_xbar_c (n)->mask & (1 << port));
  except: return 0;
}
